/* File         :    scnprtcl.h

   Description  :    Scanner communication protocol header file.

   Targets      :    1394 Protocol.

   Conditions   :    None.

   Author       :    Loc Truong.

   Copyright    :    ( c ) 1999, Identix, Inc.

   Date         :    03.29.99    Created.
 */

#ifndef SCN_PRTCL__H
#define SCN_PRTCL__H

#include  "scntype.h"

/*
** Command Codes
**
** (A '#' in the comment means there is an associated data block structure.)
*/

#define CMD_NO_OP               0       /*   No Operation                   */
#define CMD_RESET               1       /*   Reset                          */
#define CMD_SEND_VERS           5       /* # Send Version Information       */
#define CMD_SEND_HDW_VERS       6       /* # Send Hardware Version Info     */
#define CMD_SEND_IMG_VERS       7       /* # Send image processing version  */

#define CMD_SEND_IMAGE          10      /* # Send Image to Host             */
#define CMD_RECV_IMAGE          11      /* # Receive Image from Host        */
#define CMD_FREE_IMAGE          12      /* # Free Image                     */
#define CMD_GET_DRK_IMAGE       14      /* # Get Dark Reference             */ 
#define CMD_SET_DRK_IMAGE       15      /* # Set Dark Reference             */
#define CMD_GET_EQU_IMAGE       16      /* # Get Equalization Reference     */ 
#define CMD_SET_EQU_IMAGE       17      /* # Set Equalization Reference     */
#define CMD_DISPLAY_IMAGE       19      /* # Display Image In Image Area    */

#define CMD_RECV_COFF           20      /* # Receive COFF File              */
#define CMD_RECV_PCIP           21      /* # Receive PCIP Programming File  */
#define CMD_RECV_IPP            22      /* # Receive IPP Programming File   */
#define CMD_RECV_TOAD_FW        23      /* # Receive TOAD Firmware Prog File*/
#define CMD_RECV_TOAD_FPGA      24      /* # Receive TOAD FPGA Prog File    */
#define CMD_RECV_CLR_NORM       25      /* # receive color normalization tbl*/
#define CMD_RECV_KERNEL         26      /* # receive Linux kernel           */
#define CMD_RECV_RAMDISK        27      /* # receive Ram Disk               */
#define CMD_RECV_BOOTLDR        28      /* # receive Boot loader            */
#define CMD_RECV_JFFS2          29      /* # receive JFFS2 iamge            */

#define CMD_CALIB_FINGER        30      /* # Finger Scanner Calibration     */
#define CMD_CHECK_PLATEN        31      /* # Finger Platen Check            */
#define CMD_CALIB_HAND          32      /* # Hand Scanner Calibration       */
#define CMD_CHECK_ROLLER        33      /* # Hand Roller Check              */
#define CMD_CALIB_SLAP          34      /* # Slap Scanner Calibration       */
#define CMD_CHECK_FLAT          35      /* # Flat Platen Check              */
#define CMD_CALIB_PALM          36      /* # Slap Scanner Calibration       */
//#define CMD_CHECK_PALM          37      /* # Flat Platen Check              */

#define CMD_CONFIG_REMOTE_IMAGE 41      /* # Configure Remote Image Area    */
#define CMD_CNTRL_REMOTE_IMAGE  42      /* # Control Remote Image Display   */
#define CMD_CONFIG_REMOTE_MAP   43      /*   Configure Remote Mapping       */
#define CMD_CONFIG_REMOTE_BAND  44      /* # Configure Remote Band Mode/Size*/

#define CMD_SET_RESOLUTION      50      /* # Set Final Image Resolution     */
#define CMD_CONFIG_HAND_SPEED   51      /* # Config Hand Scan Speed Display */
#define CMD_CONFIG_HAND_HEIGHT  52      /* # Config Hand Scan Max Height    */
#define CMD_GET_HAND_HEIGHT     53      /* # Get Hand Scan Max Height       */
#define CMD_SET_DFC_MAP         54      /* # Set Directed Fngr Capture Map  */ 
#define CMD_GET_DFC_MAP         55      /* # Get Directed Fngr Capture Map  */ 

#define CMD_CONFIG_LOCAL_DISP   60      /* # Configure Local Display        */
#define CMD_FILL_LOCAL_DISP     61      /* # Fill Local Display             */
#define CMD_WRITE_LOCAL_DISP    62      /* # Write Local Display            */
#define CMD_CNTRL_LOCAL_IMAGE   63      /* # Control Local Image Display    */
#define CMD_CONFIG_LOCAL_PALT   64      /* # Configure Local Color Palette  */
#define CMD_CONFIG_LOCAL_MAP    65      /*   Configure Local Mapping        */
#define CMD_CONFIG_LOCAL_IMAGE  66      /* # Configure Local Image Area     */
#define CMD_CONFIG_LOCAL_OVLY   67      /* # Configure Local Image Overlay  */

#define CMD_CAPTURE_FINGER      70      /* # Capture Finger                 */
#define CMD_CAPTURE_HAND        71      /* # Capture Hand                   */
#define CMD_STOP_CAPTURE        72      /*   Stop Capture                   */
#define CMD_REVIEW_CAPTURE      73      /* # Review Capture                 */
#define CMD_CAPTURE_SLAP        74      /* # Capture Slap                   */
#define CMD_CAPTURE_PALM        75      /* # Capture Palm                   */
#define CMD_FINISH_PALM         76      /* # Finish Palm Without Capture    */
#define CMD_FINISH_FINGER       77      /* # Finish Finger Without Capture  */
#define CMD_FINISH_SLAP         78      /* # Finish Slap Without Capture    */
#define CMD_REINIT_CAPTURE_TASK 79      /*   Reinitialize Capture Task      */

#define CMD_QUERY_SWITCH        80      /* # Query Switch State             */
#define CMD_QUERY_CAPTURE       81      /*   Query Capture State            */
#define CMD_QUERY_COMMAND       82      /* # Query Command State            */
#define CMD_QUERY_IMAGE_INFO    83      /* # Query Image Information        */
#define CMD_QUERY_IMAGE_QUAL    84      /* # Query Image Quality Checks     */
#define CMD_QUERY_SCANNERS      85      /*   Query Scanner Components       */
#define CMD_QUERY_CAPTYPES      86      /*   Query Capture Types            */
#define CMD_QUERY_DEV_CAP       87      /*   Query Device Capabilities      */
#define CMD_QUERY_CAPT_CAP      88      /* # Query Capture Capabilities     */
#define CMD_QUERY_PREVIEW_INFO  89      /* # Query Info about preview image */

#define CMD_SET_PARAMETERS      90      // * Set Parameters
#define CMD_GET_PARAMETERS      91      // * Get Parameters
#define CMD_SAVE_PARAMETERS     92      //   Save Parameters to EEPROM
#define CMD_RESTORE_PARAMETERS  93      //   Restore Parameters from EEPROM
#define CMD_CHECK_PARAMETERS    94      // * Check Parameters
#define CMD_RESTORE_DEF_PARAMS  95      //   Restore Parameters from 'C' Inits
#define CMD_GET_DEFAULT_PARAMETERS 96   // * Get default parameter values

#define CMD_CONFIG_SERIAL       100     /*   Configure Serial Port          */
#define CMD_XMT_SERIAL          101     /* # Send Data to Serial Port       */

#define CMD_WRT_I2C             102     /* # Write a value to an I2C dest   */
#define CMD_READ_I2C            103     /*   Read a value from an I2C src   */
#define CMD_WRT_EEPROM          104     /* # Write a value to a eeprom loc  */
#define CMD_READ_EEPROM         105     /*   read a value from an eeprom loc*/
#define CMD_SET_LEDS	        106     /* # Set all of the LED values      */
#define CMD_GET_LEDS	        107     /* # Get all of the LED values      */

#define CMD_SAVE_HISTO          128     /* Save a histogram of a ref image  */
#define CMD_GET_HISTO           129     /* Get the saved histogram          */
#define CMD_CHECK_ILLUM         130     /* Check the illumination           */
#define CMD_CHECK_FOCUS         131     /* Check the focus                  */

#define CMD_RECV_BAK_KERNEL     140     /* # receive failsafe Linux kernel  */
#define CMD_RECV_BAK_RAMDISK    141     /* # receive failsafe Ram Disk      */
#define CMD_RECV_GUI_IMAGES     142     /* # receive GUI Images             */
#define CMD_RECV_ODB            143     /* # receive OLED driver board FW   */
#define CMD_RECV_UBL            144     /* # receive OLED driver board FW   */
 
#define CMD_TEST                200     /* execute a resident test */
/*
** Capture Types
*/
#define CAL_NOT_SUPPORTED    0x00000000
#define CAL_SUPPORTED        0x00000001
typedef enum
{
  CAP_ROLL_FNGR = 0,
  CAP_SLAP_FNGR,
  CAP_SLAP_FOUR_FNGR,
  CAP_HAND,
  CAP_PALM
} CaptureType;
   
/*
** Scanner Identification
*/
typedef enum
{
    NO_SCANNER              = 0x00,
    FINGER_SCANNER          = 0x01,
    SLAP_SCANNER            = 0x02,
    PALM_SCANNER            = 0x04,
    HAND_SCANNER            = 0x08,
    COMBO_SCANNER           = 0x10
} Scanner;

/*
** I2C Devices
*/
#define I2C_NO_SELECTION    0x0
#define I2C_RIGHT_CAMERA    0x1
#define I2C_LEFT_CAMERA     0x2
#define I2C_BOTH_CAMERAS    0x3
#define I2C_ILLUM_POT       0x4
#define I2C_IO_LEFT_HAND    0x5
#define I2C_IO_RIGHT_HAND   0x6
#define I2C_IO_THUMBS       0x7
#define I2C_IO_STATUS       0x8
#define I2C_MUX_WR          0x9
#define I2C_BOOT_EEPROM     0xa
#define I2C_LEFT_EEPROM     0xa
#define I2C_ROLL_IND        0xb
#define I2C_RIGHT_EEPROM    0xc
// The OLED device 
#define I2C_OLED_GUI        0xd

/*
** Guardian Finger 
*/
typedef struct
{
    int little;    //  Little finger 
    int ring;      //  Ring finger 
    int middle;    //  Middle finger 
    int index;     //  Index finger 
}s_leftfingers;

typedef struct
{
    int index;     //  Index finger 
    int middle;    //  Middle finger 
    int ring;      //  Ring finger 
    int little;    //  Little finger 
}s_rightfingers;

typedef struct
{
    int left;   //  Left thumb 
    int right;   //  Right thumb
} s_thumbs;

typedef struct 
{
    s_leftfingers left_four_fingers;    //  Left finger tip LEDs 
    s_rightfingers right_four_fingers;  //  Right finger tip LEDs 
    s_thumbs thumbs;               //  Thumb finger tip LEDs 
    int left_four_progress;       //  Left four finger progress LED 
    int right_four_progress;      //  Right four finger progress LED 
    int thumb_progress;           //  Thumb progress LED 
} s_LEDState;
typedef s_LEDState  Cmd_LEDState; /* for CMD_GET_LEDS CMD_SET_LEDS       */

// These assignments are determined by the hardware method of selecting
// the corresponding color. This was done to simplify the driver.
#define LED_OFF	    0
#define LED_GREEN	  1
#define LED_RED	    2
#define LED_YELLOW	3
#define LED_MASK  (LED_GREEN | LED_RED )


/*
** The LED color needs to be separated from the contact state.
*/
#define NO_CONTACT      0x100
#define GREAT_CONTACT   0x200
#define POOR_CONTACT    0x400
#define GOOD_CONTACT    0x800
#define CONTACT_MASK  ( NO_CONTACT | GREAT_CONTACT | POOR_CONTACT | GOOD_CONTACT)

typedef struct 
{
    unsigned int uiRunningSum;   /* The contact density for one frame */ 
    unsigned int uiWeight;
} s_contactInfo;

#define NO_OF_CONTACTS 5

typedef struct 
{
    s_contactInfo fingers[NO_OF_CONTACTS];		//  Contact information for up to four fingers 
}s_FingerStatus;

/*
** Structures for STA_PREVIEW_INFO
*/
/* Type of slap detected */
typedef enum{
   eSLAP_DETECTION_FNGR_IN_GAP= -5,
   eSLAP_DETECTION_NOT_IMPLEMENTED=-1,
   eSLAP_DETECTION_UNKNOWN = 0,
   eSLAP_DETECTION_LEFT_HAND,
   eSLAP_DETECTION_RIGHT_HAND,
   eSLAP_DETECTION_TWO_THUMBS,
   eSLAP_DETECTION_SINGLE_FNGR
}e_SlapDetected;

/* Measurement of the contact using density*/
typedef enum
{
   eNO_CONTACT,
   ePOOR_CONTACT,
   eGOOD_CONTACT,
   eGREAT_CONTACT
}e_FngrContact;

#if 1
typedef enum

{
   eRTQA_TOO_LIGHT,
   eRTQA_TOO_DARK,
   eRTQA_SMEARED,
   eRTQA_GOOD
} e_FngrRtqaApp;

/* All of the information for a single finger */
typedef struct
{
   e_FngrRtqaApp  eFngrRtqa;
   e_FngrContact  eContact;		//  Contact 
   U32            uiGoodPct;
}s_PreviewFngrInfo;

#else
/* App RTQA numbers */
typedef struct
{
  U32 uiContactPct;  /* percent of the platen area with contact        */
  U32 uiGoodPct;     /* percent of the contact area with good cells    */
  U32 uiDarkPct;     /* percent of the contact area with dark cells    */
  U32 uiLightPct;    /* percent of the contact area with light cells   */
  U32 uiSmearPct;    /* percent of the contact area with smeared cells */
}s_FngrRtqaApp;

/* All of the information for a single finger */
typedef struct
{
   s_FngrRtqaApp  sFngrRtqa;
   e_FngrContact  eContact;		//  Contact 
   U32            uiGoodPct;
}s_PreviewFngrInfo;
#endif

/*
** Structure to report all available information about the current raw image
** Associated with CMD_QUERY_PREVIEW_INFO
*/
#define FNGR_ARR_SIZE 4
typedef struct
{
   s_PreviewFngrInfo sFngr[FNGR_ARR_SIZE];
   e_SlapDetected eSlapDetected;
}Sta_PreviewFngrInfo;

typedef struct 
{
    int imageType;        //  left four, right four, thumbs
    int stableTime;       //  time in ms that image must be stable before capture is executed
//    struct QAThresholds;  //  QA thresholds that must be met before image is considered "good"
}s_AutoCaptureParms;

/*
** Image Data Types
*/
typedef enum
{
    RAW_FNGR = 1,       /*  Raw finger image (raw resolution)               */
    RAW_SLAP,           /*  Raw slap image (raw resolution)                 */
    RAW_HAND,           /*  Raw hand image (raw resolution)                 */
    QUAL_FNGR,          /*  Finger quality analysis image (raw resolution)  */
    QUAL_SLAP,          /*  Slap quality analysis image (raw resolution)    */
    QUAL_HAND,          /*  Hand quality analysis image (raw resolution)    */
    FINAL_FNGR,         /*  Final finger image (customer resolution)        */
    FINAL_SLAP,         /*  Final slap image (customer resolution)          */
    FINAL_HAND,         /*  Final hand image (customer resolution)          */
    MINU_TMPLT,         /*  Minutiae template                               */
    EQ_IMAGE,           /*  Equalization image                              */
    RIGHT_1ST,          /*  Right four 1st finger                           */
    RIGHT_2ND,          /*  Right four 2nd finger                           */
    RIGHT_3RD,          /*  Right four 3rd finger                           */
    RIGHT_4TH,          /*  Right four 4th finger                           */
    LEFT_1ST,           /*  Left four 1st finger                            */
    LEFT_2ND,           /*  Left four 2st finger                            */
    LEFT_3RD,           /*  Left four 3st finger                            */
    LEFT_4TH,           /*  Left four 4st finger                            */
   EQ_IMAGE_MOD,        /*  Modified image for use with Micron cameras      */
   DARKENED_FNGR,       /*  Darkened Raw Finger Image                       */
   DARKENED_SLAP,       /*  Darkened Raw Slap Image                         */
   DISPLAY_IMAGE,       /*  Display Image                                */
    RAW_PALM,           /*  Raw palm image (raw resolution)                 */
    QUAL_PALM,          /*  Palm quality analysis image (raw resolution)  */
    FINAL_PALM,         /*  Final palm image (customer resolution)        */
   ALT_SLAP_LAT,        /*  Previously stored slap latent image */
   ALT_FNGR_LAT,        /*  Previously stored finger latent image           */
   DARK_REF,            /*  Dark Reference Image - Paladin imager           */
   DECIMATED,           /*  Reference Image - decimated                     */
   PREVIEW_EQU,         /*  The actively decimated reference image used for
                            equalizing the current decimated image          */
   ROLL_EDGES           /*  Holds the edge array when rolling fingers       */
} Image_DataType;


/*
** Image Scan Types
*/
typedef enum
{
    NO_SCAN_TYPE,
    RIGHT_THUMB = 1,
    RIGHT_INDEX,
    RIGHT_MIDDLE,
    RIGHT_RING,
    RIGHT_LITTLE,
    LEFT_THUMB,
    LEFT_INDEX,
    LEFT_MIDDLE,
    LEFT_RING,
    LEFT_LITTLE,
    PLAIN_RIGHT_THUMB,
    PLAIN_RIGHT_INDEX,
    PLAIN_RIGHT_MIDDLE,
    PLAIN_RIGHT_RING,
    PLAIN_RIGHT_LITTLE,
    PLAIN_LEFT_THUMB,
    PLAIN_LEFT_INDEX,
    PLAIN_LEFT_MIDDLE,
    PLAIN_LEFT_RING,
    PLAIN_LEFT_LITTLE,
    RIGHT_FOUR,
    LEFT_FOUR,
    RIGHT_FOUR_1ST,
    RIGHT_FOUR_2ND,
    RIGHT_FOUR_3RD,
    RIGHT_FOUR_4TH,
    LEFT_FOUR_1ST,
    LEFT_FOUR_2ND,
    LEFT_FOUR_3RD,
    LEFT_FOUR_4TH,
    RIGHT_HAND,
    LEFT_HAND,
    RIGHT_PALM,
    LEFT_PALM,
    RIGHT_WRITERS_PALM,
    LEFT_WRITERS_PALM,
    BOTH_THUMBS,

    PLATEN_CALIB,
    FLAT_CALIB,
    ROLLER_CALIB,
    PALM_CALIB,
    SINGLE_FNGR,
    USE_DFC_MAP
} Image_ScanType;          

//-------------------------------------------------
//            DFC Finger Map Bits
//-------------------------------------------------
#define BIT_RTHUMB      0x0001
#define BIT_RINDEX      0x0002
#define BIT_RMIDDLE     0x0004
#define BIT_RRING       0x0008
#define BIT_RLITTLE     0x0010
#define BIT_LTHUMB      0x0020
#define BIT_LINDEX      0x0040
#define BIT_LMIDDLE     0x0080
#define BIT_LRING       0x0100
#define BIT_LLITTLE     0x0200

//-------------------------------------------------
//            DFC Defines for convenience
//-------------------------------------------------
#define DFC_ENABLE      0x8000
#define DFC_EMPTY       0x0000

//-------------------------------------------------
//            SCANNER Finger Map Bits
//            conforms to internal array structures
//-------------------------------------------------
#define RLITTLE_ARR_BIT     0x0200
#define RRING_ARR_BIT       0x0100
#define RMIDDLE_ARR_BIT     0x0080
#define RINDEX_ARR_BIT      0x0040
#define RTHUMB_ARR_BIT      0x0020
#define LTHUMB_ARR_BIT      0x0010
#define LINDEX_ARR_BIT      0x0008
#define LMIDDLE_ARR_BIT     0x0004
#define LRING_ARR_BIT       0x0002
#define LLITTLE_ARR_BIT     0x0001

/*--------------------------------------------------------------------------*/
/*
** Special Image Tags for Equalization Reference Images
*/
/*  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  */

#define   EQU_TAG_FINGER        ((U32)-2)
#define   EQU_TAG_SLAP          ((U32)-3)
#define   EQU_TAG_HAND          ((U32)-4)
#define   EQU_TAG_PALM          ((U32)-5)

/*
** Finger Calibration Modes
**
** These flags specify modes and actions for finger scanner calibration
** and platen check.
*/
typedef enum
{
    FNGR_CMODE_SAVE_RAW     = 0x00000001,   /* (same as FNGR_SMODE_SAVE_RAW)*/
    FNGR_CMODE_FILTER       = 0x00000002,   /* filter calibration image     */
    FNGR_CMODE_REPLACE      = 0x00000004,   /* replace reference image      */
    FNGR_CMODE_DARK_REF     = 0x00000008,   /* this is a dark reference img */
    FNGR_CMODE_TO_FLASH     = 0x00000010,   /* save image to local flash    */
    FNGR_CMODE_CAMERA_B     = 0x00000020,   /* select camera b              */
    FNGR_CMODE_COLOR_NRMLZ  = 0x00000040,   /* select Normalization         */
    FNGR_CMODE_DARK_SUB     = 0x00000080,   /* subtract the dark image      */
    FNGR_CMODE_STATE_MSGS   = 0x01000000,   /* send capture state messages  */
    FNGR_CMODE_MOTION_MSGS  = 0x02000000,   /* send capt prog motion mesgs  */
    FNGR_CMODE_DISABLE_KEYS = 0x08000000,   /* Disable the scan/save keys   */
    FNGR_CMODE_DISABLE_BEEP = 0x80000000    /* Disable the automatic beep   */
} Finger_CalibMode;          


/*
** Finger Scan Modes
**
** These flags specify modes and actions for fingerprint capture.
*/
typedef enum
{
    FNGR_SMODE_PLAIN        = 0x00000000,   /* capture plain fingerprint    */
    FNGR_SMODE_SAVE_RAW     = 0x00000001,   /* raw image will be retained   */
    FNGR_SMODE_ROLL_RIGHT   = 0x00000002,   /* capture rolled fingerprint   */
    FNGR_SMODE_ROLL_LEFT    = 0x00000004,   /*   (constrained to right or   */
    FNGR_SMODE_ROLL_EITHER  = 0x00000006,   /*   left or unconstrained)     */
    FNGR_SMODE_EQUALIZE     = 0x00000008,   /* perform scanner equalization */
    FNGR_SMODE_PREVIEW      = 0x00000010,   /* preview scan before capture  */
    FNGR_SMODE_CAMERA_B     = 0x00000020,   /* select camera b              */
    FNGR_SMODE_COLOR_NRMLZ  = 0x00000040,   /* select Normalization         */
    FNGR_SMODE_DARK_SUB     = 0x00000080,   /* subtract the dark image      */
    FNGR_SMODE_TIP_SMEAR    = 0x00000100,   /* perform tip-smear correction */
    FNGR_SMODE_CLEAN_BGND   = 0x00000200,   /* correct for background latent*/
    FNGR_SMODE_PREVIEW_INFO = 0x00000400,   /* generate info on preview     */
    FNGR_SMODE_COVERAGE_QA  = 0x00001000,   /* measure print coverage factor*/
    FNGR_SMODE_IMAGE_QA     = 0x00002000,   /* measure image quality factors*/
    FNGR_SMODE_AUDIT        = 0x00004000,   /* require host audit of print  */
    FNGR_SMODE_REVIEW       = 0x00008000,   /* require operator review      */
    FNGR_SMODE_TRANSFORM    = 0x00010000,   /* transform to final resolution*/
    FNGR_SMODE_CONTRAST     = 0x00020000,   /* adjust final contrast        */
    FNGR_SMODE_SHARPEN      = 0x00040000,   /* sharpen final image          */
    FNGR_SMODE_UI           = 0x00080000,   /* enable the resident UI       */
    FNGR_SMODE_AUTOCAP_NOPRE= 0x00800000,   /* Auto capture no preview */
    FNGR_SMODE_STATE_MSGS   = 0x01000000,   /* send capture state messages  */
    FNGR_SMODE_MOTION_MSGS  = 0x02000000,   /* send capt prog motion mesgs  */
    FNGR_SMODE_AUTOCAP_PRE  = 0x04000000,   /* Auto capture with preview */
    FNGR_SMODE_DISABLE_KEYS = 0x08000000,   /* Disable the scan/save keys   */
    FNGR_SMODE_SPECIAL_DISP = 0x70000000,   /* display special info (local) */
    FNGR_SMODE_FREEZE_OVLY  = 0x40000000,   /*    freeze curve overlay      */
    FNGR_SMODE_CONTACT_OVLY = 0x50000000,   /*    contact area overlay      */
    FNGR_SMODE_CONTACT_OVWR = 0x60000000,   /*    contact area overwrite    */
    FNGR_SMODE_EDGES_OVWR   = 0x70000000,   /*    filtered edges overwrite  */
    FNGR_SMODE_DISABLE_BEEP = 0x80000000    /* Disable the automatic beep   */
} Finger_ScanMode;          
// labels shortened to align the bit assignments
#define FNGR_SMODE_AUTOCAP_NOPREVIEW FNGR_SMODE_AUTOCAP_NOPRE
#define FNGR_SMODE_AUTOCAP_PREVIEW FNGR_SMODE_AUTOCAP_PRE

/*
** These bits do double duty for testing purposes. 
*/
#define  FNGR_SMODE_NOISE_TST    FNGR_SMODE_ROLL_RIGHT /* Test noise fltr   */
#define  FNGR_SMODE_SHARP_TST    FNGR_SMODE_ROLL_LEFT  /* Test sharpen fltr */

/*
** Slap Calibration Modes
**
** These flags specify modes and actions for slap scanner calibration and
** platen check.
*/
typedef enum
{
    SLAP_CMODE_SCAN         = 0x00000000,   /* calibrate scan mode          */
    SLAP_CMODE_SAVE_RAW     = 0x00000001,   /* (same as SLAP_SMODE_SAVE_RAW)*/
    SLAP_CMODE_FILTER       = 0x00000002,   /* filter calibration image     */
    SLAP_CMODE_REPLACE      = 0x00000004,   /* replace reference image      */
    SLAP_CMODE_DARK_REF     = 0x00000008,   /* this is a dark reference img */
    SLAP_CMODE_TO_FLASH     = 0x00000010,   /* save image to local flash    */
    SLAP_CMODE_COLOR_NRMLZ  = 0x00000040,   /* select Normalization         */
    SLAP_CMODE_DARK_SUB     = 0x00000080,   /* subtract the dark image      */
    SLAP_CMODE_STATE_MSGS   = 0x01000000,   /* send capture state messages  */
    SLAP_CMODE_MOTION_MSGS  = 0x02000000,   /* send capt prog motion mesgs  */
    SLAP_CMODE_DISABLE_KEYS = 0x08000000,   /* Disable the scan/save keys   */
    SLAP_CMODE_ENABLE_SUB   = 0x10000000,   /* Enable the sub-sampling during live preview   */
    SLAP_CMODE_DISABLE_BEEP = 0x80000000    /* Disable the automatic beep   */
} Slap_CalibMode;          


/*
** Slap Scan Modes
**
** These flags specify modes and actions for four-finger slap print capture.
*/
typedef enum
{
    SLAP_SMODE_SAVE_RAW     = 0x00000001,   /* raw image will be retained   */
    SLAP_SMODE_EQUALIZE     = 0x00000008,   /* perform scanner equalization */
    SLAP_SMODE_PREVIEW      = 0x00000010,   /* preview scan before capture  */
    SLAP_SMODE_COLOR_NRMLZ  = 0x00000040,   /* select Normalization         */
    SLAP_SMODE_DARK_SUB     = 0x00000080,   /* subtract the dark image      */
    SLAP_SMODE_CLEAN_BGND   = 0x00000200,   /* correct for background latent*/
    SLAP_SMODE_PREVIEW_INFO = 0x00000400,   /* generate info on preview     */
    SLAP_SMODE_IMAGE_QA     = 0x00002000,   /* measure image quality factors*/
    SLAP_SMODE_AUDIT        = 0x00004000,   /* require host audit of print  */
    SLAP_SMODE_REVIEW       = 0x00008000,   /* require operator review      */
    SLAP_SMODE_TRANSFORM    = 0x00010000,   /* transform to final resolution*/
    SLAP_SMODE_CONTRAST     = 0x00020000,   /* adjust final contrast        */
    SLAP_SMODE_SHARPEN      = 0x00040000,   /* sharpen final image          */
    SLAP_SMODE_UI           = 0x00080000,   /* enable the resident UI       */
    SLAP_SMODE_RIGHT        = 0x00100000,   /* Update right hand info       */
    SLAP_SMODE_CENTER       = 0x00200000,   /* Update center hand info      */
    SLAP_SMODE_LEFT         = 0x00400000,   /* Update left hand info        */
    SLAP_SMODE_AUTOCAP_NOPRE= 0x00800000,   /* Auto capture no preview */
    SLAP_SMODE_STATE_MSGS   = 0x01000000,   /* send capture state messages  */
    SLAP_SMODE_MOTION_MSGS  = 0x02000000,   /* send capt prog motion mesgs  */
    SLAP_SMODE_AUTOCAP_PRE  = 0x04000000,   /* Auto capture with preview */
    SLAP_SMODE_DISABLE_KEYS = 0x08000000,   /* Disable the scan/save keys   */
    SLAP_SMODE_ENABLE_SUB   = 0x10000000,   /* TP4101 Only, disables on TP5300
                                               and TP4800 */
    SLAP_SMODE_DISABLE_BEEP = 0x80000000    /* Disable the automatic beep   */
} Slap_ScanMode;          
// labels shortened to align the bit assignments
#define SLAP_SMODE_AUTOCAP_NOPREVIEW SLAP_SMODE_AUTOCAP_NOPRE
#define SLAP_SMODE_AUTOCAP_PREVIEW SLAP_SMODE_AUTOCAP_PRE

/*
** Hand Calibration Modes
**
** These flags specify modes and actions for hand scanner calibration and
** roller check.
*/
typedef enum
{
    HAND_CMODE_SAVE_RAW     = 0x00000001,   /* (same as FNGR_SMODE_SAVE_RAW)*/
    HAND_CMODE_FILTER       = 0x00000002,   /* filter calibration image     */
    HAND_CMODE_REPLACE      = 0x00000004,   /* replace reference image      */
    HAND_CMODE_STATE_MSGS   = 0x01000000,   /* send capture state messages  */
    HAND_CMODE_MOTION_MSGS  = 0x02000000,   /* send capt prog motion mesgs  */
    HAND_CMODE_DISABLE_KEYS = 0x08000000,   /* Disable the scan/save keys   */
    HAND_CMODE_DISABLE_BEEP = 0x80000000    /* Disable the automatic beep   */
} Hand_CalibMode;          


/*
** Hand Scan Modes
**
** These flags specify modes and actions for hand print capture.
*/
typedef enum
{
    HAND_SMODE_SAVE_RAW     = 0x00000001,   /* raw image will be retained   */
    HAND_SMODE_EQUALIZE     = 0x00000008,   /* perform scanner equalization */
    HAND_SMODE_PREVIEW      = 0x00000010,   /* preview scan before capture  */
    HAND_SMODE_XFORM_LINES  = 0x00000040,   /* transform horiz. resolution  */
    HAND_SMODE_CLEAN_BGND   = 0x00000200,   /* correct for background latent*/
    HAND_SMODE_COVERAGE_QA  = 0x00001000,   /* measure print coverage factor*/
    HAND_SMODE_IMAGE_QA     = 0x00002000,   /* measure image quality factors*/
    HAND_SMODE_AUDIT        = 0x00004000,   /* require host audit of print  */
    HAND_SMODE_REVIEW       = 0x00008000,   /* require operator review      */
    HAND_SMODE_TRANSFORM    = 0x00010000,   /* transform vertical resolution*/
    HAND_SMODE_CONTRAST     = 0x00020000,   /* adjust final contrast        */
    HAND_SMODE_SHARPEN      = 0x00040000,   /* sharpen final image          */
    HAND_SMODE_EXTRACT      = 0x00100000,   /* extract separate fingerprints*/
    HAND_SMODE_MINUTIA      = 0x00200000,   /* extract minutia information  */
    HAND_SMODE_STATE_MSGS   = 0x01000000,   /* send capture state messages  */
    HAND_SMODE_MOTION_MSGS  = 0x02000000,   /* send capt prog motion mesgs  */
    HAND_SMODE_DISABLE_KEYS = 0x08000000,   /* Disable the scan/save keys   */
    HAND_SMODE_SPEED_DISP   = 0x10000000,   /* overlay speed info (display) */
    HAND_SMODE_DISABLE_BEEP = 0x80000000    /* Disable the automatic beep   */
} Hand_ScanMode;          


/*
** Palm Calibration Modes
**
** These flags specify modes and actions for palm scanner calibration and
** platen check.
*/
typedef enum
{
    PALM_CMODE_SCAN         = 0x00000000,   /* calibrate scan mode          */
    PALM_CMODE_SAVE_RAW     = 0x00000001,   /* (same as PALM_SMODE_SAVE_RAW)*/
    PALM_CMODE_FILTER       = 0x00000002,   /* filter calibration image     */
    PALM_CMODE_REPLACE      = 0x00000004,   /* replace reference image      */
    PALM_CMODE_DARK_REF     = 0x00000008,   /* this is a dark reference img */
    PALM_CMODE_TO_FLASH     = 0x00000010,   /* save image to local flash    */
    //PALM_CMODE_COLOR_NRMLZ  = 0x00000040,   /* select Normalization         */
    PALM_CMODE_DARK_SUB     = 0x00000080,   /* subtract the dark image      */
    PALM_CMODE_STATE_MSGS   = 0x01000000,   /* send capture state messages  */
    PALM_CMODE_MOTION_MSGS  = 0x02000000,   /* send capt prog motion mesgs  */
    PALM_CMODE_DISABLE_KEYS = 0x08000000,   /* Disable the scan/save keys   */
    PALM_CMODE_DISABLE_BEEP = 0x80000000    /* Disable the automatic beep   */
} Palm_CalibMode;          


/*
** Palm Scan Modes
**
** These flags specify modes and actions for palm slap print capture.
*/
typedef enum
{
    PALM_SMODE_SAVE_RAW     = 0x00000001,   /* raw image will be retained   */
    PALM_SMODE_EQUALIZE     = 0x00000008,   /* perform scanner equalization */
    PALM_SMODE_PREVIEW      = 0x00000010,   /* preview scan before capture  */
    //PALM_SMODE_COLOR_NRMLZ  = 0x00000040,   /* select Normalization         */
    PALM_SMODE_DARK_SUB     = 0x00000080,   /* subtract the dark image      */
    PALM_SMODE_CLEAN_BGND   = 0x00000200,   /* correct for background latent*/
    PALM_SMODE_PREVIEW_INFO = 0x00000400,   /* generate info on preview     */
    PALM_SMODE_IMAGE_QA     = 0x00002000,   /* measure image quality factors*/
    PALM_SMODE_AUDIT        = 0x00004000,   /* require host audit of print  */
    PALM_SMODE_REVIEW       = 0x00008000,   /* require operator review      */
    PALM_SMODE_TRANSFORM    = 0x00010000,   /* transform to final resolution*/
    PALM_SMODE_CONTRAST     = 0x00020000,   /* adjust final contrast        */
    PALM_SMODE_SHARPEN      = 0x00040000,   /* sharpen final image          */
    PALM_SMODE_AUTOCAP_NOPRE= 0x00800000,   /* Auto capture no preview */
    PALM_SMODE_STATE_MSGS   = 0x01000000,   /* send capture state messages  */
    PALM_SMODE_MOTION_MSGS  = 0x02000000,   /* send capt prog motion mesgs  */
    PALM_SMODE_AUTOCAP_PRE  = 0x04000000,   /* Auto capture with preview */
    PALM_SMODE_DISABLE_KEYS = 0x08000000,   /* Disable the scan/save keys   */
    PALM_SMODE_DISABLE_SUB  = 0x10000000,   /* Disable the sub-sampling during live preview   */
    PALM_SMODE_DISABLE_BEEP = 0x80000000    /* Disable the automatic beep   */
} Palm_ScanMode;          
// labels shortened to align the bit assignments
#define PALM_SMODE_AUTOCAP_NOPREVIEW PALM_SMODE_AUTOCAP_NOPRE
#define PALM_SMODE_AUTOCAP_PREVIEW PALM_SMODE_AUTOCAP_PRE

/*
** Units of Resolution
*/
typedef enum
{
    DOTS_PER_INCH,
    DOTS_PER_100MM
} Resolution_Units;


/*
** Local Display Screen Modes
*/
typedef enum
{
    LOC_DISP_OFF,
    LOC_DISP_1024x768x8
} LocDisp_ScreenMode;


/*
** Local Display Image Modes
**
** NOTE: With _IMAGE_DISABLE the host completely controls the local display.
**       The host should use this mode when it is copying a screen saver
**       bitmap to the local display.
*/
typedef enum
{
    DISP_IMAGE_DISABLE,     /* do not allow display of prints in image area */
    DISP_IMAGE_ENABLE       /* allow display of prints in image area        */
} Disp_ImageMode;


/*
** Command Data Blocks
**
** These structures provide additional information required with various
** commands.
*/

typedef struct          /* for CMD_SEND_IMAGE                               */
{
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U32 image_type;     /* image data type (see Image_DataType)             */
    U32 size;           /* xfer size <= actual image size (width * height)  */
} Cmd_SendImagData;


typedef struct          /* for CMD_RECV_IMAGE                               */
{
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U32 image_type;     /* image data type (see Image_DataType)             */
    U32 height;         /* height                                           */
    U32 width;          /* width                                            */
    U32 pitch;          /* pitch                                            */
    U32 vertl_res;      /* vertical resolution                              */
    U32 hzntl_res;      /* horizontal resolution                            */
    U32 size;           /* xfer size <= actual image size (width * height)  */
} Cmd_RecvImagData;


typedef struct          /* for CMD_FREE_IMAGE                               */
{
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (specify 0 to delete all images  */
                        /*    associated with handle)                       */
    U32 image_type;     /* image data type (specify 0 to delete all images  */
                        /*    associated with handle and scan type)         */
} Cmd_FreeImagData;


typedef struct          /* for CMD_GET_EQU_IMAGE and CMD_SET_EQU_IMAGE      */
{
    U32 handle;         /* image handle                                     */
    U32 scanner;        /* type of scanner (see Scanner)                    */
} Cmd_EquImagData;


typedef struct          /* for CMD_DISPLAY_IMAGE                            */
{
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U32 image_type;     /* image data type (see Image_DataType)             */
    U32 repetitions;    /* number of times to display image                 */
    U32 quality;        /* non-zero for a quality mapped image              */
    U32 offset;         /* initial vertical offset                          */
    U32 white;          /* white background fill level (when offset != 0)   */
} Cmd_DispImag;


typedef struct          /* for CMD_RECV_COFF, CMD_RECV_PCIP,                */
{                       /*     and CMD_RECV_IPP                             */
    U32 size;           /* coff file data size                              */
} Cmd_RecvCoffData;


typedef struct          /* for CMD_CALIB_FINGER, CMD_CHECK_PLATEN,          */
{                       /*     CMD_CALIB_HAND, CMD_CHECK_ROLLER,            */
                        /*     CMD_CALIB_SLAP, and CMD_CHECK_FLAT,          */
                        /*     CMD_CALIB_PALM                               */
    U32 handle;         /* assigned image handle                            */
    U32 scan_mode;      /* scan mode (see {Finger,Slap,Hand}_CalibMode)     */
    U32 frames;         /* number of frames to accumulate                   */
} Cmd_CalibData;


typedef struct          /* for CMD_GRB_TEST_MODE                            */
{
    U32 test_mode;      /* grabber test mode (see Grb_TestMode)             */
    U32 test_value;     /* 9-bit value substituted for camera data          */
} Cmd_GrbTestMode;


typedef struct          /* for CMD_CONFIG_REMOTE_IMAGE                      */
{
    U32 hzntl_ext;      /* width of image area (multiple of four)           */
    U32 vertl_ext;      /* height of image area                             */
} Cmd_ConfigRemImage;

typedef struct          /* for CMD_CONFIG_HAND_HEIGHT                      */
{
    U32 height;         /* height of image area in 1000ths 1 inch == 1000                            */
} Cmd_ConfigMaxHandHgt;


typedef struct          /* for CMD_CNTRL_REMOTE_IMAGE                       */
{
    U32 mode;           /* image display mode (see Disp_ImageMode)          */
} Cmd_CntrlRemImage;


typedef struct          /* for CMD_CONFIG_REMOTE_BAND                       */
{
    U32 lines;          /* number of lines per band (0 for full frame)      */
} Cmd_ConfigRemBand;


typedef struct          /* for CMD_CONFIG_LOCAL_DISP                        */
{
    U32 display_mode;   /* local display mode (see LocDisp_ScreenMode)      */
} Cmd_ConfigLocDisp;


typedef struct          /* for CMD_FILL_LOCAL_DISP                          */
{
    U32 hzntl_pos;      /* horizontal position (multiple of four)           */
    U32 vertl_pos;      /* vertical position                                */
    U32 hzntl_ext;      /* width (multiple of four)                         */
    U32 vertl_ext;      /* height                                           */
    U32 pixel;          /* fill pixel value                                 */
} Cmd_FillDispData;


typedef struct          /* for CMD_WRITE_LOCAL_DISP                         */
{
    U32 hzntl_pos;      /* horizontal position (multiple of four)           */
    U32 vertl_pos;      /* vertical position                                */
    U32 hzntl_ext;      /* width (multiple of four)                         */
    U32 vertl_ext;      /* height                                           */
} Cmd_WriteDispData;


typedef struct          /* for CMD_CNTRL_LOCAL_IMAGE                        */
{
    U32 mode;           /* image display mode (see Disp_ImageMode)          */
} Cmd_CntrlLocImage;


typedef struct          /* for CMD_CONFIG_LOCAL_PALT                        */
{
    U32 size;           /* number of palette entries to read from data chnnl*/
} Cmd_ConfigLocPalt;


typedef struct          /* for CMD_CONFIG_LOCAL_IMAGE                       */
{
    U32 hzntl_pos;      /* horizontal position (multiple of four)           */
    U32 vertl_pos;      /* vertical position                                */
    U32 hzntl_ext;      /* width (multiple of four)                         */
    U32 vertl_ext;      /* height                                           */
} Cmd_ConfigLocImage;


typedef struct          /* for CMD_CONFIG_LOCAL_OVLY                        */
{
    U32 color;          /* overlay transparency color in image area         */
} Cmd_ConfigLocOvly;


typedef struct          /* for CMD_SET/GET_DFC_MAP                          */
{
    U32 scan_fngrs;     /* bitmapped finger array                           */
} Cmd_DfcMap;


typedef struct          /* for CMD_SET_RESOLUTION                           */
{
    U32 scanner;        /* type of scanner (see Scanner)                    */
    U32 units;          /* units of resolution (see Resolution_Units)       */
    U32 resolution;     /* resolution specification                         */
} Cmd_SetResolution;


typedef struct          /* for CMD_CONFIG_HAND_SPEED                        */
{
    U32 left;           /* width of left indicator (0-33 % image width)     */
    U32 center;         /* width of center indicator (0-33 % image width)   */
    U32 right;          /* width of right indicator (0-33 % image width)    */
} Cmd_ConfigHandSpeed;


typedef struct          /* for CMD_CAPTURE_FINGER, CMD_CAPTURE_HAND,        */
{                       /*     CMD_CAPTURE_SLAP, CMD_CAPTURE_PALM,          */
                        /*     CMD_FINISH_FINGER, CMD_FINISH_SLAP,          */
                        /*     CMD_FINISH_PALM                              */
    U32 handle;         /* assigned image handle                            */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U32 scan_mode;      /* scan mode (see {Finger,Slap,Hand}_ScanMode)      */
} Cmd_StartCaptData;


typedef struct          /* for CMD_REVIEW_CAPTURE                           */
{
    U32 choice;         /* 0 - allow only rejection; 1 - operator's choice  */
} Cmd_ReviewCapt;

#define  SW_BEEP_ONLY           3
#define  SW_RETURN_IMMEDIATELY  2
#define  SW_CLEAR_OLD_STATE     1
#define  SW_NO_CLEAR            0
#define  SW_CANCEL             -1

typedef struct          /* for CMD_QUERY_SWITCH                             */
{
    S32 mode;           /* -1: cancel a previous query switch command       */
                        /*  0: check switches without clearing prior state  */
                        /*  1: clear prior state before checking switches   */
                        /*  2: return switches immediately (no wait/prompt) */
    U32 timeout;        /* wait timeout period (msec; 0: wait indefinitely) */
    U32 prompt;         /* 0: no audible prompt; 1-3: no. of audible tones  */
    U32 freq[3];        /* frequency of up to 3 audible tones (in Hertz)    */
    U32 time[3];        /* duration of up to 3 audible tones (in msec)      */
} Cmd_QuerySwitch;


typedef struct          /* for CMD_QUERY_COMMAND                            */
{
    U32 tag_id;         /* querying command's tag ID                        */
} Cmd_QueryCmdData;


typedef struct          /* for CMD_QUERY_IMAGE_INFO                         */
{
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U32 image_type;     /* image data type (see Image_DataType)             */
} Cmd_QueryImage;


typedef struct          /* for CMD_QUERY_IMAGE_QUAL                         */
{
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
} Cmd_QueryQuality;

typedef struct          /* for CMD_QUERY_PREVIEW_INFO                       */

{
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
} Cmd_QueryPreviewInfo;


typedef struct          /* for CMD_SET_PARAMETERS, CMD_GET_PARAMETERS,      */
{                       /*     and CMD_CHECK_PARAMETERS                     */
    U32 parameter_set;  /* code (see Parameter_Set)                         */
    U32 parameter_size; /* size in bytes of parameter info                  */
} Cmd_SetParameters;


typedef struct          /* for CMD_XMT_SERIAL                               */
{
    U32 ser_port;       /* serial channel 0 or 1                            */
    U32 ser_count;      /* number of bytes of data to send                  */
    U8  ser_data [32];  /* data to be sent                                  */
} Cmd_XmtSerial;

typedef struct          /* for CMD_WRT_I2C, CMD_READ_I2C                    */
{
    U8  ucDest;         /* camera selection                                 */
    U16 usRegister;     /* Register address                                 */
    U16 usValue;        /* Register Value                                   */
} Cmd_I2C;

typedef struct          /* for CMD_WRT_EEPROM, CMD_READ_EEPROM              */
{
    U16 usAddress;      /* physical address within the EEPROM               */
    U8  ucValue;        /* Value to be written                              */
} Cmd_Eeprom;

#define ABORT_CAPTURE     0
#define SCAN_KEY_PRESSED  1
#define SAVE_KEY_PRESSED  2
typedef struct          /* for CMD_STOP_CAPTURE                           */
{
  U32 uiChoice;         /* 0 - abort capture;  Scan key pressed           */
} Cmd_StopCapt;


typedef struct          /* for CMD_QUERY_CAPT_CAP                        */
{
    U32 captureType;    /* what capture type to return capabilities for  */
} Cmd_QueryCaptCap;

#define GUI_OLED_FLASH_PAGE   4096
typedef struct
{
    U32 uiFlashAddr;
	U8  ucImage[GUI_OLED_FLASH_PAGE];
} GUI_Image;

typedef struct      /* for CMD_RECV_GUI_IMAGES                                   */
{
    U32 uiNumImages;    /* number of images                                      */
	//GUI_Image images[]; /* variable number of GUI_Image structures             */
} Cmd_RecvGUIImages;

typedef struct      /* for CMD_TEST                                   */
{
   U32 uiTestType;
   U32 uiTestOptions;
} Cmd_Test;

/*
**  Union of All Command Data Blocks and Basic Command Block
*/
typedef union
{
   Cmd_SendImagData     send_image;
   Cmd_RecvImagData     recv_image;
   Cmd_FreeImagData     free_image;
   Cmd_EquImagData      equ_image;
   Cmd_DispImag         disp_image;
   Cmd_RecvCoffData     recv_coff;
   Cmd_GrbTestMode      grb_test;
   Cmd_ConfigRemImage   cfg_rimg;
   Cmd_CntrlRemImage    cntrl_rimg;
   Cmd_ConfigRemBand    cfg_rband;
   Cmd_ConfigLocDisp    cfg_ldisp;
   Cmd_FillDispData     fill_ldisp;
   Cmd_WriteDispData    write_ldisp;
   Cmd_CntrlLocImage    cntrl_limg;
   Cmd_ConfigLocPalt    cfg_lpalt;
   Cmd_ConfigLocImage   cfg_limg;
   Cmd_ConfigLocOvly    cfg_lovly;
   Cmd_CalibData        img_calib;
   Cmd_SetResolution    set_res;
   Cmd_ConfigHandSpeed  cfg_hspd;
   Cmd_StartCaptData    img_capt;
   Cmd_ReviewCapt       rvw_capt;
   Cmd_QuerySwitch      qry_switch;
   Cmd_QueryCmdData     qry_cmd;
   Cmd_QueryImage       qry_img;
   Cmd_QueryQuality     qry_qual;
   Cmd_SetParameters    set_parm;
   Cmd_XmtSerial        xmt_serial;
   Cmd_I2C              wrt_I2C;
   Cmd_I2C              read_I2C;
   Cmd_Eeprom           wrt_eeprom;
   Cmd_Eeprom           rd_eeprom;
   Cmd_LEDState         sLEDState;
   Cmd_StopCapt         sStopCapture;
   Cmd_QueryCaptCap     qry_capt_cap;
   Cmd_QueryPreviewInfo qry_prev_info;
   Cmd_ConfigMaxHandHgt cfg_hand;
   Cmd_Test             canned_test;
   Cmd_DfcMap           sDfcMap;
} Cmd_DataBlock;


typedef struct
{
   S32  code;               /* command code                                 */
   U32  tag_id;             /* unique tag id; 0 = no tag                    */
   U32  data_size;          /* size of command data block                   */
} Cmd_CmdBasic;


typedef struct
{
   Cmd_CmdBasic     cmd;    /* command's basic block                        */
   Cmd_DataBlock    data;   /* command's data block                         */
} Cmd_CommandPlusData;



/*
** Status/Event Codes
**
** (An '#' in the comment means there is an associated data block structure.)
*/

#define STA_NO_OP               1       /*   NO_OP Response                 */
#define STA_RESET               2       /*   Scanner Reset                  */
#define STA_SEND_VERS           4       /* # Send Versions                  */
#define STA_SEND_HDW_VERS       6       /* # Send Hardware Versions         */
#define STA_SEND_IMG_VERS       7       /* # Send Image Processing Versions */
#define STA_DFC_MASK            8       /* # Send Local copy of DFC mask    */

#define STA_BUFF_OVERFLOW       12      /* # Buffer Overflow                */

#define STA_CHNL_XFER_ERR       14      /* # Channel Transfer/Protocol Error*/
#define STA_CHNL_SEND_RESP_FAIL 15      /* # Channel Send Status Resp Error */

#define STA_IMG_XFER_CONFIRM    26      /* # IMG Transfer Confirmation      */
#define STA_DAT_XFER_CONFIRM    27      /* # DAT Transfer Confirmation      */
#define STA_BMP_XFER_CONFIRM    28      /* # BMP Transfer Confirmation      */

#define STA_PLATEN_CHECK        30      /* # Finger Scan Platen Check Result*/
#define STA_ROLLER_CHECK        31      /* # Hand Roller Check Result       */
#define STA_FLAT_CHECK          32      /* # Slap Scan Platen Check Result  */
#define STA_ILLUM_CHECK         33      /* # Check illumination Result      */
#define STA_FOCUS_CHECK         34      /* # Result of Focus check          */

#define STA_FINALIZED           51      /* # Final Image Available          */
#define STA_IMAGE_INFO          52      /* # Image Size Information         */
#define STA_IMAGE_QUAL          53      /* # Image Quality Information      */

#define STA_QUERY_CAPTYPES      80      /* # Report capture characteristics */
#define STA_QUERY_DEV_CAP       81      /* # Report Device Capabilities     */
#define STA_QUERY_CAPT_CAP      82      /* # Report Capture Capabilities    */
#define STA_PREVIEW_INFO        83      /* # Report info about preview image*/
#define STA_MAX_HAND_SIZE       84      /* # Report info max hand size      */
 
#define STA_READ_CAMERA         103     /*   value read from a camera reg   */
#define STA_READ_EEPROM         105     /*   value read from an eeprom loc  */
#define STA_READ_LEDS           106     /*   value read from LEDs           */

#define STA_SWITCH_STATE        130     /*   Switch State                   */
#define STA_CAPTURE_STATE       131     /* # Capture State                  */
#define STA_CAPTURE_PROGRESS    132     /* # Capture Progress               */
#define STA_COMMAND_STATUS      133     /*   Command Status                 */
#define STA_SCANNER_STATUS      134     /*   Scanner System Status          */
#define STA_SCANNERS_PRESENT    135     /* # Scanner Components Present     */

#define STA_RCV_SERIAL          141     /* # Data Received from Serial Port */


/*
** Basic State Codes
**
** The following define codes assigned to the basic state member for various
** status/event codes.
*/

typedef struct              // for STA_QUERY_CAPTYPES
{
  struct s_capInfo
  {
    U32          calibration;    // CMD_CALIB_XXX support
    U32          cur_resolution;
    U32          imgMilsW;       // image width (in mils)
    U32          imgMilsH;       // image height (in mils)
    U32          imgIsochMaxW;   // width of largest real time image
    U32          imgIsochMaxH;   // height of largest real time image
  } capType[4];                  // in CaptureType order
} Sta_QryCaps;

typedef enum    /* for STA_RESET, STA_BUFF_OVERFLOW, and STA_SCANNER_STATUS */
{
    SYSTEM_OK   =  0,   /* Non-Error system event                           */
    COMM_FAIL   = -1,   /* Other Comm. error (from 1394 SW stack)           */
    RTOS_FAIL   = -2,   /* RTOS related error                               */
    MEM_ERROR   = -3,   /* Memory (allocation, free) error                  */
    APP_ERROR   = -4    /* Application related (system resouce) error       */
} Sta_SysFailStatus;


typedef enum    /* for STA_SWITCH_STATE                                     */
{
    BSCAN_LATCHED   = 0x01, /* scan button pressed since last switch check  */
    BSAVE_LATCHED   = 0x02, /* save button pressed since last switch check  */
    FSAVE_LATCHED   = 0x04, /* save foot-switch pressed since last check    */
    FSCAN_LATCHED   = 0x08, /* scan foot-switch pressed since last check    */
    NOT_BSCAN_NOW   = 0x10, /* scan button current state (0 - pressed)      */
    NOT_BSAVE_NOW   = 0x20, /* save button current state (0 - pressed)      */
    NOT_FSAVE_NOW   = 0x40, /* save foot-switch current state (0 - pressed) */
    NOT_FSCAN_NOW   = 0x80, /* scan foot-switch current state (0 - pressed) */
    SWITCH_TIMEOUT  = 0xF0
} Sta_SwitchState;


typedef enum    /* for STA_COMMAND_STATUS, STA_FINALIZED, STA_PLATEN_CHECK, */
{               /*     STA_ROLLER_CHECK, STA_FLAT_CHECK, and STA_NO_OP      */
    CMD_SUCCESS =  3,   /* execution successful                             */
    CMD_BUSY    =  2,   /* submitted for execution                          */
    CMD_PENDING =  1,   /* pending execution                                */
    CMD_UNKNOWN =  0,   /* status unknown or irrelevant                     */
    CMD_REJECT  = -1,   /* command rejected (COP busy)                      */
    CMD_CANCEL  = -2,   /* aborted by explicit request                      */
    CMD_ERROR   = -3,   /* command data error (invalid or                   */
                        /* corrupted data/parameter(s)                      */
    CMD_DATA    = -4,   /* data/buffer/resource not readily available       */
                        /* for command setup and/or execution               */
    CMD_FAIL    = -5,   /* execution fail due to other abnormalities        */
    CMD_FAIL_HS = -6,   /* greater than maximum hand size                   */
    CMD_DFC_TOOMANY  = -7,   /* too many fingers set in DFC map                  */
    CMD_DFC_BADCOMBO = -8    /* invalid combination of fingers set in DFC map    */
} Sta_CmdRespStatus;


/*
** Auxiliary State Codes
**
** The following are assigned to state members inside data blocks associated
** with various status/event codes.
*/

typedef enum    /* for STA_IMG_XFER_CONFIRM, STA_DAT_XFER_CONFIRM,          */
{               /*      and STA_BMP_XFER_CONFIRM                            */
    XFER_READY = 1,
    XFER_BUSY,
    XFER_ERROR,
    XFER_ABORT,
    XFER_DONE
} Xfer_State;


typedef enum    /* for STA_IMG_XFER_CONFIRM, STA_DAT_XFER_CONFIRM,          */
{               /*      and STA_BMP_XFER_CONFIRM                            */
    CHNL_NO_REQUEST,    /* no request                                       */
    CHNL_RECV_REQST,    /* reqeusted reception -- waiting for buffer        */
    CHNL_SEND_REQST,    /* requested transmission -- waiting for buffer     */
    CHNL_RECV_READY,    /* reception buffer ready                           */
    CHNL_SEND_READY     /* tranmission buffer ready                         */
} Buffer_Cntrl;

/*
** Xfer Response Token Field Masks
**
** for STA_IMG_XFER_CONFIRM, STA_DAT_XFER_CONFIRM, and STA_BMP_XFER_CONFIRM
*/
#define RSP_FAIL                0x80000000  /* failure                      */
#define RSP_PROG_FIRST          (1 << 30)   /* first response block         */
#define RSP_PROG_LAST           (1 << 29)   /* last response block          */
#define RSP_COUNT_BITS          0x0000FFFF  /* byte count or failure code   */

/*
** Xfer Response Token Failure Codes  (when RSP_FAIL is set)
**
** for STA_IMG_XFER_CONFIRM, STA_DAT_XFER_CONFIRM, and STA_BMP_XFER_CONFIRM
*/
#define RSP_NO_RESULT       0       /* xfer result not available            */
#define RSP_SEND_PREVIOUS   1       /* re-send previous transaction         */
#define RSP_INVALID_REQST   2       /* invalid request token                */
#define RSP_INVALID_SEQNC   3       /* invalid sequence number              */
#define RSP_CHECKSUM_ERR    4       /* checksum error                       */
#define RSP_BUFF_SIZE_ERR   5       /* requested buffer size error          */
#define RSP_CHANNEL_BUSY    6       /* busy - channel not ready             */
#define RSP_XFER_BUFF_ERR   7       /* xfer buffer error - 1394 mblock      */
#define RSP_DATA_BUFF_ERR   8       /* data buffer full or unavailable      */
#define RSP_COMM_SYNC_ERR   9       /* channel comm. is out of sync.        */
#define RSP_READ_FAILURE    10      /* read failure                         */
#define RSP_WRITE_FAILURE   11      /* write failure                        */
#define RSP_TIMEOUT_ERROR   12      /* no ack/response available - timeout  */
#define RSP_APPLICAT_ERR    13      /* application error                    */


typedef enum    /* for STA_CAPTURE_STATE                                    */
{
    UNAVAILABLE,    /* major failure prevents any capture                   */
    AVAILABLE,      /* available to start new capture                       */
    PREVIEW,        /* viewing preparatory to capture                       */
    CAPTURE,        /* capturing requested print                            */
    ANALYSIS,       /* undergoing quality checks                            */
    INQUIRY,        /* waiting for host inquiries                           */
    DECISION,       /* awaiting operator/host decision on image             */
    QUERYSWITCH,    /* querying switch state                                */
    PALM_LAT_CAPT,  /* Capturing a palm latent                              */
    CLEANING_IMAGE, /* Cleaning captured image                              */
    CALIBRATION,    /* Processing calibration image                         */
    FINISHING_IMAGE /* Finishing Image                                      */
} Sta_CaptStateCode;


typedef enum    /* for STA_CAPTURE_PROGRESS                                 */
{
    ABSENT,         /* nothing in contact with scanner                      */
    PRESSING,       /* contact detected and increasing (finger scan)        */
    LIFTING,        /* contact detected and decreasing (finger scan)        */
    STEADY,         /* contact detected w/out area change (finger scan)     */
    ROLLING_LEFT,   /* contact area moving leftward (finger scan)           */
    ROLLING_RIGHT,  /* contact area moving rightward (finger scan)          */
    TOUCHING,       /* contact detected (hand scan)                         */
    SPINNING_UP,    /* scanner cylinder is turning upward (hand scan)       */
    SPINNING_DOWN,  /* scanner cylinder is turning downward (hand scan)     */
    ABORTED,        /* capture aborted by operator                          */
    STARTED,        /* capture started                                      */
    REFUSED_FAST,   /* finger/Hand moved too fast                           */
    REFUSED_SLOW,   /* finger/Hand moved too slow                           */
    CAPTURED,       /* raw image is available                               */
    ANALYZED,       /* decorated QA image is available                      */
    REJECTED,       /* captured image rejected                              */
    ACCEPTED,       /* captured image accepted                              */
    CANCELED,       /* capture canceled by Host (Stop Capture CMD)          */
    FAILED,         /* capture failed due to hardware or system errors      */

//NEW Values
    UNAVAILABLE_ISOC_1, /* ISOC ERR Could not allocate           */
    UNAVAILABLE_ISOC_2, /* ISOC ERR Could not start              */
    UNAVAILABLE_ISOC_3, /* ISOC ERR Could not stop               */
    UNAVAILABLE_IPP,    /* IPP ERR                               */
    UNAVAILABLE_DISP,   /* Display Could not allocate memory     */
    UNAVAILABLE_LATENT, /* Latent Could not  allocate memory     */   
    UNAVAILABLE_LINE,   /* Capture line too large                */    
    UNAVAILABLE_ABORT   /* Abort Command sentarge                */    
} Sta_CaptProgCode;


/*
** Status Data Blocks
**
** These structures provide additional information with various status/event
** codes.
*/

#define MAX_VERSION_STRING 12
typedef struct{
  unsigned char sVers[MAX_VERSION_STRING];    //null terminated string
  unsigned char sPartNo[MAX_VERSION_STRING];  //null terminated string
} HdwVersion;

typedef struct      /* for STA_SEND_HDW_VERS                                    */
{
  HdwVersion sFirmWareVers;
  HdwVersion uiPciVers;
  HdwVersion sIppVers;
  HdwVersion sPldVers;
  HdwVersion sBootRomVers;
  HdwVersion sCamerBdVers1;
  HdwVersion sCamerBdVers2;
} Sta_SendHdwVersData;
  
typedef struct      /* for STA_SEND_VERS                                    */
{
    char assy_make[8];  /* assembly make (e.g. "IDX")                       */
    char assy_model[16];/* assembly model (e.g. "TP-2000")                  */
    char assy_sn[16];   /* assembly serial number (e.g. "123456")           */
    U16  assy_year;     /* assembly date - year (e.g. 2001)                 */
    U8   assy_month;    /* assembly date - month (1-12)                     */
    U8   assy_day;      /* assembly date - day (1-31)                       */
    char soft_version[12];/* software version (e.g. "1.01")                 */
    U16  soft_year;     /* software build date - year (e.g. 2001)           */
    U8   soft_month;    /* software build date - month (1-12)               */
    U8   soft_day;      /* software build date - day (1-31)                 */
} Sta_SendVersData;

typedef struct      /* for STA_SEND_IMG_VERS                                    */
{
    char version[12];   /* software version (e.g. "1.01")                 */
} Sta_SendImgVersData;


typedef struct      /* for STA_BUFF_OVERFLOW                                */
{
    U32 buf_id;         /* buffer ID = module ID + buffer # (w/in module)   */
    U32 buf_size;       /* buffer size/count                                */
} Sta_BuffStatData;


typedef struct      /* for STA_MAX_HAND_HEIGHT */
{
    U32 height;      /* height                                     */
} Sta_MaxHandHgt;

typedef struct      /* for STA_CHNL_XFER_ERR and STA_CHNL_SEND_RESP_FAIL    */
{
    U32 chnl;           /* channel type                                     */
} Sta_ChnlCommErrData;


typedef struct      /* for STA_IMG_XFER_CONFIRM                             */
{
    U32 xfer_stat;      /* xfer state (see Xfer_State)                      */
    U32 xfer_type;      /* xfer buffer type (see Buffer_Cntrl)              */
    S32 status;         /* xfer status (xfer response token)                */
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type                                  */
    U32 image_type;
    U32 seq_count;      /* transferred sequence count                       */
    U32 byte_count;     /* transferred byte count                           */
} Sta_ImgXferData;


typedef struct      /* for STA_DAT_XFER_CONFIRM and STA_BMP_XFER_CONFIRM    */
{
    U32 xfer_stat;      /* xfer state (see Xfer_State)                      */
    U32 xfer_type;      /* xfer buffer type (see Buffer_Cntrl)              */
    S32 status;         /* xfer result (xfer response token)                */
    U32 seq_count;      /* transferred sequence count                       */
    U32 byte_count;     /* transferred byte count                           */
} Sta_XferCnfm;


typedef struct      /* for STA_PLATEN_CHECK and STA_ROLLER_CHECK            */
{                   /*     and STA_FLAT_CHECK                               */
    U32 tag_id;         /* command tag of original capture command          */
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U32 total_pixels;   /* total number of pixels checked                   */
    U32 dirty_pixels;   /* number of dirty pixels found                     */
    U32 damaged_pixels; /* number of damaged pixels found                   */
    U32 peak_level;     /* most prevalent illumination level                */
} Sta_CheckResult;


typedef struct      /* for STA_FINALIZED                                    */
{
    U32 tag_id;         /* command tag of original capture command          */
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
} Sta_Finalized;


typedef struct      /* for STA_IMAGE_INFO                                   */
{
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U32 img_type;       /* image data type (see Image_DataType)             */
    U32 height;         /* number of lines high                             */
    U32 width;          /* number of active pixels per line                 */
    U32 pitch;          /* number of bytes per line                         */
    U32 depth;          /* number of bits per pixel                         */
    U32 v_res;          /* vertical resolution                              */
    U32 h_res;          /* horizontal resolution                            */
} Sta_ImgInfo;


typedef struct      /* for STA_IMAGE_QUAL                                   */
/*
** NOTE: The cover_factor and roll_time will be zero for plain fingerprints
**       or if xxx_SMODE_COVERAGE_QA was not requested.  All members after
**       roll_time will be zero if xxx_IMAGE_QA was not requested (see
**       Finger_ScanMode, Slap_ScanMode, and Hand_ScanMode).
*/
{
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U32 img_type;       /* image data type (see Image_DataType)             */
    U16 cover_factor;   /* roll coverage factor (times 100)                 */
    U16 roll_time;      /* number of milliseconds during roll               */
    U16 contact_area;   /* % area of contact                                */
    U16 good_area;      /* % area of good contrast                          */
    U16 dark_area;      /* % area too dark                                  */
    U16 light_area;     /* % area too light                                 */
    U16 smear_area;     /* % area too smeared                               */
} Sta_ImgQual;


typedef struct      /* for STA_IMAGE_QUAL_EXT                               */
/*
*/
{
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U16 contact_area;   /* % area of contact                                */
    U16 good_area;      /* % area of good contrast                          */
    U16 dark_area;      /* % area too dark                                  */
    U16 light_area;     /* % area too light                                 */
    U16 smear_area;     /* % area too smeared                               */
} s_ImgQual;


typedef struct      /* for STA_IMAGE_QUAL_EXT                                   */
/*
*/
{
    U32 handle;            /* image handle                                  */
    U32 scan_type;         /* image scan type (see Image_ScanType)          */
    U32 img_type;          /* image data type (see Image_DataType)          */
    s_ImgQual sImgQual[4]; /* image quality information                     */
} Sta_ImgQualExt;


typedef struct      /* for STA_CAPTURE_STATE and STA_CAPTURE_PROGRESS       */
{
    U32 tag_id;         /* command tag of original capture command          */
    U32 handle;         /* image handle                                     */
    U32 scan_type;      /* image scan type (see Image_ScanType)             */
    U32 code;           /* state or progress code                           */
                        /*   (see Sta_CaptStateCode and Sta_CaptProgCode)   */
    s_FingerStatus sFingerStatus;
} Sta_CaptInfo;

 
typedef struct      /* for STA_SCANNERS_PRESENT                             */
{
    U32 count;          /* number of scanner components present (up to 6)   */
    struct
    {
        U8  id;             /* identification (see Scanner)                 */
        U8  version;        /* version (0-255)                              */
        U16 serno;          /* serial number (0-65535)                      */
        U16 year;           /* mfg. date - year (e.g., 2002)                */
        U8  month;          /* mfg. date - month (1-12)                     */
        U8  day;            /* mfg. date - day (1-31)                       */
    } scanners[6];
} Sta_Scanners;

 
typedef struct      /* for STA_RCV_SERIAL                                   */
{
    U32 ser_port;       /* serial channel 0 or 1                            */
    U32 ser_count;      /* bytes received                                   */
    U8  ser_data [32];  /* data block                                       */
} Sta_RcvSerial;

typedef struct          /* for STA_ILLUM_CHECK                              */
{
  U32 uiNewPixelSum; /* Sum of all the pixel values in new ref image        */
  U32 uiOldPixelSum; /* Stored sum of the original ref image                */
  U8  ucScore;       /* Measurement of how well the two images matched 
                        1 - bad, 100 - best */
} Sta_CheckIllum;

typedef struct          /* for STA_FOCUS_CHECK                              */
{
  U8  ucScore;       /* Measurement of how well the two images matched 
                        1 - bad, 100 - best */
} Sta_CheckFocus;

typedef s_LEDState Sta_LEDState;  /* for STA_READ_LEDS                       */

typedef struct      /* for setting maximum size                             */
{
    U8  dummy [108];
} Sta_MaxSize;

typedef struct          /* for STA_DFC_MAP                          */
{
    U32 scan_fngrs;     /* bitmapped finger array                           */
} Sta_DfcMap;

/*
**  Union of All Status/Event Data Blocks and Basic Status/Event Block
*/

typedef union
{
    Sta_SendImgVersData sVerinfo;
    Sta_SendVersData    ver_info;
    Sta_BuffStatData    buf_stat;
    Sta_ChnlCommErrData chnl_err;
    Sta_ImgXferData     xfer_img;
    Sta_XferCnfm        xfer_cnfm;
    Sta_CheckResult     check;
    Sta_Finalized       final;
    Sta_ImgInfo         img_info;
    Sta_ImgQual         img_qual;
    Sta_ImgQualExt      img_qualext;
    Sta_CaptInfo        capt_info;
    Sta_Scanners        scanners;
    Sta_RcvSerial       rcv_serial;
    Sta_QryCaps         sta_qrycaps;
    Sta_MaxSize         max_size;
    Sta_LEDState        sLEDState;
    Sta_CheckFocus      sCheckFocus;
    Sta_CheckIllum      sCheckIllum;
    Sta_SendHdwVersData hdw_info;
    Sta_PreviewFngrInfo sPreviewFngrInfo;
    Sta_MaxHandHgt      hand_size;
    Sta_DfcMap          sDfcMap;
} Sta_DataBlock;


typedef struct
{
    U32 code;               /* status/event code                            */
    U32 type;               /* UNUSED                                       */
    S32 state;              /* basic state code                             */
    U32 tag_id;             /* unique tag id; 0 = no tag                    */
    U32 data_size;          /* size of status/event's data block            */
} Sta_EventBasic;

typedef struct
{
    Sta_EventBasic  event;  /* status/event's basic block                   */
    Sta_DataBlock   data;   /* status/event's data block                    */
} Sta_EventPlusData;


/*
** Capability Related for STA_QUERY_DEV_CAP
*/
// CPB Tag | length | Data if applicable
// CPB Tag is 2 bytes in size
// length is 2 bytes in size
// length is length of data in bytes

#define DEV_CPB_UI_TYPES				1       /* Device supports these UI type(s)     */
#define DEV_CPB_UI_TYPES_SIZE			1       /* Size of CPB_UI_TYPES in bytes        */

// for CPB_UI_TYPES
typedef enum
{
    UI_TYPE_NONE		    = 1,
    UI_TYPE_4_4_2_TRI_LED,                /* 4,4,2 tri-colored LED */
    UI_TYPE_4_4_2_OLED,                   /* 4,4,2 OLED */
    UI_TYPE_4_4_2_MONO_LED                /* 4,4,2 single LED */
} UI_TYPES;          

/*
** Capability Related for STA_QUERY_CAPT_CAP
*/
// CPB Tag | length | Data if applicable
// CPB Tag is 2 bytes in size
// length is 2 bytes in size
// length is length of data in bytes
#define CT_CPB_CALIBRATION				1       /* CMD_CALIB_XXX support                */
#define CT_CPB_CUR_RESOLUTION			2
#define CT_CPB_CUR_IMG_MILS_W			3       /* image width (in mils)                */
#define CT_CPB_CUR_IMG_MILS_H			4       /* image height (in mils)               */
#define CT_CPB_CUR_IMG_ISOCH_MAX_W	5       /* width of largest real time image     */
#define CT_CPB_CUR_IMG_ISOCH_MAX_H  6       /* height of largest real time image    */
#define CT_CPB_AUTO_CAP					7       /* Device supports auto capture         */
#define CT_CPB_FINISHING			   8       /* Device supports finishing        */
#define CT_CPB_CLEANING 			   9       /* Device supports cleaning         */
#define CT_CPB_RTQA      			   10      /* Device supports realtime RTQA    */
#define CT_CPB_RT_DEC   			   11      /* Device supports realtime 
                                               decorated image */
#define CT_CPB_SLAPDET              12      /* detects slap type                */
#define CT_CPB_EQUALIZE             13      /* Support equalization full image  */
#define CT_CPB_FAST_PREVIEW         14      /* Support the use of a smalle imatge
                                               during preview */
#define CT_CPB_DECIMATION           15     /*  Decimation of the display image
                                               is not supported. */
#define CT_CPB_DARK_IMAGE_COMP      16      /* performs dark image compensation */
#define CT_CPB_DFC                  17      /* Directed finger capture */

#define CT_CPB_CALIBRATION_SIZE     1       /* unsigned char                    */
#define CT_CPB_RESOLUTION_SIZE      2       /* unsigned short                   */
#define CT_CPB_CUR_IMG_MILS_W_SIZE  2       /* unsigned short                   */
#define CT_CPB_CUR_IMG_MILS_H_SIZE  2       /* unsigned short                   */
#define CT_CPB_CUR_IMG_ISOCH_W_SIZE 2       /* unsigned short                   */
#define CT_CPB_CUR_IMG_ISOCH_H_SIZE 2       /* unsigned short                   */
#define CT_CPB_AUTO_CAP_SIZE        1       /* unsigned char                    */

#define CT_CPB_FINISHING_SIZE		   1
#define CT_CPB_CLEANING_SIZE		   1
#define CT_CPB_RTQA_SIZE  			   1
#define CT_CPB_RT_DEC_SIZE			   1
#define CT_CPB_SLAPDET_SIZE		   1
#define CT_CPB_EQUALIZE_SIZE		   1
#define CT_CPB_FAST_PREVIEW_SIZE	   1
#define CT_CPB_DECIMATION_SIZE      1
#define CT_CPB_DARK_IMAGE_SIZE      1
#define CT_CPB_DFC_SIZE             2
 
#define CAPABILITY_SUPPORTED			1       /* Device supports these UI type(s)     */
#define CAPABILITY_NOT_SUPPORTED		0       /* Size of CPB_UI_TYPES in bytes        */

/*
** Declarations concerning resident tests within the scanner.
*/ 
typedef enum
{
   eLoopTest = 0, 
   ePaladin_Camera,     // Paladin camera test
   eTiger_IPP_SDRAM     // Tiger board IPP SDRAM
} eTestTypes;

typedef enum
{
   eTiger_Bank0 = 0,        //Select Tiger IPP SDRAM bank 0
   eTiger_Bank1, 
   eTiger_Bank2,
   eTiger_Bank3,
   eTiger_BankAll
} eTigerIPP_Options;

typedef enum
{
   eTest_55 = 0,          //Paladin camera 
   eTest_AA,          //Paladin camera 
   eTest_Normal,
   eTest_SensorPat1, // ??
   eTest_SensorPat2, // ??
   eTest_RampUp,      //Paladin camera 
   eTest_RampDown     //Paladin camera 
}eCameraTestOptions;

#define TEST_OPTION_INCR     0

#endif
